/* $Id: vtlio.c,v 1.85 2000/02/15 19:25:37 yujik Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Keith Bayern */

/* Routines for low level I/O to E1432 module, using VTL */

#ifdef HAVE_VTL  /* this file is only used for VTL */

#define GTL printf ("got to line %ld in file %s\n", __LINE__, __FILE__);

/* the next two defines try to fix a temporary NI block transfer problem */
#if 0
#define MAX_NI_BLOCK 256 	/* limits block size for viMove() */
#define FIX_NI_VIMOVE 		/* break up transfers into chunks */
#endif

extern long pnptracelevel; /* from hpe1432.c */

/* maximum number of open sessions */
#ifdef WIN32
#define MAX_VIOPENS 30
#else
#define MAX_VIOPENS 10
#endif

#include "sema.h"
#include "hpe1432.h"

/* the default session for VTL */
static ViSession	viSession_drm;	 /* default resource manager */

/* array of structs used by i1432_visa_setup() */
static long NumOpenViSessions; /* total boards viOpened */
static struct {
	ViSession sessionID; /* session returned by viOpen*/
	ViBoolean open;     /* true if this sessionID/LA in currently open */
	} openSessions[256];


static SHORTSIZ16 convert_error_VTL_to_SICL(ViStatus verr)
{
SHORTSIZ16 serr;

  serr=0;
  if(verr < VI_SUCCESS)
  {
    switch(verr)
    {
    case VI_ERROR_INV_OBJECT:
    case VI_ERROR_INV_OFFSET:
    case VI_ERROR_RSRC_LOCKED:

    case VI_ERROR_NSUP_OPER:
    case VI_ERROR_INV_SPACE:
    case VI_ERROR_NSUP_WIDTH:
    default:
	serr=ERR1432_UNABLE_TO_MAP_MEMORY;
	break;

    case VI_ERROR_BERR:
	serr=ERR1432_BUS_ERROR;
	break;

    case VI_ERROR_NSUP_OFFSET:
	serr=ERR1432_ILLEGAL_REGISTER_OFFSET;
	break;
	
    }
  }
  return serr;
}

/* 
 *
 * after a call to this i1432_module_connect, the module_node is ready to 
 * be used to communicate to the E143x
 */
SHORTSIZ16 i1432_module_connect(E1432_MODULE_LIST_NODE *mn)
{
static     char    errstr[80]; /* static for VTL error message */
long	i;
char    str[50];
ViStatus	vierr;
ViSession	visess;

#if 0
if(NumOpenViSessions<0)
    printf("vtlio.c:i1432_module_connect, NumOpenViSessions=%ld\n",
	       (long)NumOpenViSessions);

printf("entering i1432_module_connect for la=%ld\n",(long)mn->la);
#endif
    /* check sanity of the mn */
    if((mn->la>255)||(mn->la<1))
    {
        i1432_set_error_info("bad LA in i1432_module_connect");
	return ERR1432_UNABLE_TO_OPEN_SICL;
    }
    /* check if this LA is already open */
    if(openSessions[mn->la].open)
    {
	mn->sicl_id=openSessions[mn->la].sessionID;
	return 0;
    }

#if 0
printf("entering i1432_module_connect for la=%ld\n",(long)mn->la);
printf("not open, need to open, NumOpenViSessions= %ld\n",(long)NumOpenViSessions);
#endif

    /* check if too many opens, and close one */
    if(NumOpenViSessions>MAX_VIOPENS)
    {

#if 0
printf("looking for open session\n");
#endif
	for(i=1;i<256;i++)
	{
	    if(openSessions[i].open)
	    {
		/* found one, close it */

		if (i1432_visa_lock_mode)
		    /* Unlock the session first */
		    (void) viUnlock(openSessions[i].sessionID);

#if 0
printf("closing la%ld\n",(long)i);
#endif
		viClose(openSessions[i].sessionID);
		openSessions[i].sessionID=0;
		openSessions[i].open=0;
		NumOpenViSessions--;
		break;
	    }
	}
    }

    /* open VTL session for card */
    (void) sprintf(str, "%s::%d::INSTR",i1432_sicl_name,mn->la);
#if 0
printf("calling viOpen with %s\n",str);
#endif
    vierr = viOpen(viSession_drm,str,VI_NULL,VI_NULL,&visess);
#if 0
printf("viOpen returned %ld\n",vierr);
#endif
    if (vierr != VI_SUCCESS) {
        i1432_set_error_info("viOpen failed in i1432_visa_open");
	(void) sprintf(errstr,"viOpen returned error %ld",(long)vierr);
    	i1432_error_info = errstr;
	return i1432_la_print_error(mn->la, ERR1432_UNABLE_TO_OPEN_SICL);
    }
    openSessions[mn->la].sessionID = (long)visess;
    openSessions[mn->la].open = 1;
    mn->sicl_id = (long)visess;

    /* count this opening */
    NumOpenViSessions++;

    if (i1432_visa_lock_mode)
    {
	/* The user has requested locking.  Try to lock, but don't
	   wait if we can't do the lock. */
	vierr = viLock(visess, VI_EXCLUSIVE_LOCK, VI_TMO_IMMEDIATE,
		       VI_NULL, VI_NULL);
	if (vierr != VI_SUCCESS &&
	    vierr != VI_SUCCESS_NESTED_EXCLUSIVE &&
	    vierr != VI_SUCCESS_NESTED_SHARED)
	    return i1432_la_print_error(mn->la, ERR1432_LOCK_FAILED);
    }

    if (i1432_visa_introff_count > 0)
    {
	vierr = viEnableEvent(visess, VI_ALL_ENABLED_EVENTS,
			      VI_SUSPEND_HNDLR, VI_NULL);
	/* NI VISA on HP-UX returns invalid event if no events
	   are currently handled, so allow
	   VI_ERROR_INV_EVENT.  HP VISA on HP-UX returns
	   VI_ERROR_HANDLER_NINSTALLED in the same situation,
	   so allow that too. */
	if (vierr < VI_SUCCESS &&
	    vierr != VI_ERROR_INV_EVENT &&
	    vierr != VI_ERROR_HNDLR_NINSTALLED)
	    return i1432_print_error(convert_error_VTL_to_SICL(vierr));
    }

    return 0;
}

SHORTSIZ16
i1432_visa_get_lock_mode(int *mode)
{
    *mode = i1432_visa_lock_mode;
    return 0;
}

SHORTSIZ16
i1432_visa_set_lock_mode(int mode)
{
    ViStatus vierr;
    int     i;

    if (mode && !i1432_visa_lock_mode)
	/* Lock openned sessions, which were previously unlocked */
	for (i = 1; i < 256; i++)
	    if (openSessions[i].open)
	    {
		/* Try to lock, but don't wait if we can't do the lock. */
		vierr = viLock(openSessions[i].sessionID,
			       VI_EXCLUSIVE_LOCK, VI_TMO_IMMEDIATE,
			       VI_NULL, VI_NULL);
		if (vierr != VI_SUCCESS &&
		    vierr != VI_SUCCESS_NESTED_EXCLUSIVE &&
		    vierr != VI_SUCCESS_NESTED_SHARED)
		    return i1432_la_print_error((SHORTSIZ16) i,
						ERR1432_LOCK_FAILED);
	    }

    if (!mode && i1432_visa_lock_mode)
	/* Unlock open sessions, which were previously locked */
	for (i = 1; i < 256; i++)
	    if (openSessions[i].open)
		(void) viUnlock(openSessions[i].sessionID);

    i1432_visa_lock_mode = mode;

    return 0;
}

SHORTSIZ16 EXPORT
e1432_init_io_driver(void)
{
    int     i;
    ViStatus	vierr;

    /* If previously initialized, uninitialize, then reinitialize */
    if (i1432_lib_init)
	(void) e1432_uninit_io_driver();

    /* open the default resource manager session */
    vierr=viOpenDefaultRM(&viSession_drm);
    if(vierr!=VI_SUCCESS)
    {
        i1432_set_error_info("viOpenDefaultRM failed");
	return ERR1432_UNABLE_TO_OPEN_SICL;
    }


    /* reset visa open manager */
    NumOpenViSessions=0;
    for(i=0;i<256;i++)
    {
        openSessions[i].open=0;
    }



#ifdef	HAVE_SIGBUS
    i1432_buserr_trap = 0;
    i1432_buserr_env_valid = 0;
#endif
    i1432_lib_init = 1;
    i1432_mod_list = 0;
    i1432_group_list = NULL;
    i1432_trace_level = 0;
    i1432_diag_print_level = 0;
    i1432_print_reg_access = 0;
    i1432_print_errors = 0;
    i1432_error_info = NULL;
    i1432_error_str = "";
    for (i = 0; i < E1432_CHAN_TYPES; i++)
	i1432_chan_list[i] = NULL;

    i1432_dnld_fw_state = 0;   /* substrate FW not downloaded */
    i1432_dnld_fw_err = 0;     /* no substrate FW download error */
    i1432_install_file = NULL; /* no download file selected */
    i1432_trigger_sent = 0;
    i1432_kludge_data_rpm_flag = 0;
    i1432_kludge_data_rpm_id1 = 0;
    i1432_kludge_data_rpm1 = 0;
    i1432_kludge_data_rpm2 = 0;
    i1432_kludge_enable_flag = 0;
    i1432_kludge_tach_flag = 0;
    i1432_rs_irq_error = 0;
    i1432_visa_d16 = 0;
    i1432_visa_introff_count = 0;
    i1432_visa_lock_mode = 0;
    return 0;
}

SHORTSIZ16 EXPORT
e1432_uninit_io_driver(void)
{
    SHORTSIZ16 error;

    /* Free any memory in channel groups */
    error = e1432_delete_all_chan_groups(NULL);
    if (error)
	return error;

    /* Free any memory in channel and module lists, unmap imapped
       memory, close SICL device sessions. */
    error = e1432_assign_channel_numbers(0, NULL, NULL);
    if (error)
	return error;

    /* close default resource manager */
    if(viSession_drm)
    {
        viClose(viSession_drm);
	viSession_drm=0;
    }

    i1432_lib_init = 0;

    return 0;
}


/* return true if E1432 at description */
static int vxiident(char *s)
{
ViStatus	vierr;
ViSession	vi;
ViUInt16	idreg;
ViUInt16	dtype;
int	rv;

/* this gets run before i1432_init_io_library, so it must call visa directly */

    rv=0;
    vierr=viOpen(viSession_drm,s,VI_NULL,VI_NULL,&vi);
    if(vierr!=VI_SUCCESS)
	return 0;

    vierr=viIn16(vi,VI_A16_SPACE,E1432_ID_REG,&idreg);
    if(vierr!=VI_SUCCESS)
    {
	viClose(vi);
	return 0;
    }
    
    vierr=viIn16(vi,VI_A16_SPACE,E1432_DEVICE_TYPE_REG,&dtype);
    if(vierr!=VI_SUCCESS)
    {
	viClose(vi);
	return 0;
    }
    
#define SEMA_MODEL_CODE         (0x0200)
#define DTYP_MODEL_CODE_MASK    (0x0FF8)
#define SUBC_MFR_ID_MASK        (0x0FFF)
#define SUBC_MFR_HP             (0x0FFF)

    if(   ((idreg&SUBC_MFR_ID_MASK)==SUBC_MFR_HP)
	&&((dtype&DTYP_MODEL_CODE_MASK)==SEMA_MODEL_CODE))
	  rv=1;


    viClose(vi);
    return rv;

}

/* extract LA from descript */
static LONGSIZ32 getLA(char *s)
{
LONGSIZ32 rv;
char *cp;


    rv=0;
    cp=strpbrk(s,":");
    if(cp)
    {
        if(*cp==':')
        {
	   cp++;
           if(*cp==':')
	   {
	       cp++;
	       rv=atoi(cp);
	   }
        }
    }
    return rv;
}

int i1432_find_modules(LONGSIZ32 *LAlist,LONGSIZ32 listSize,LONGSIZ32 *numFound,
		 char *rsrc,LONGSIZ32 rsrcLen)
{
LONGSIZ32 found;
ViChar		buffer[VI_FIND_BUFLEN];
ViRsrc		descrip=buffer;
ViUInt32	num;
ViFindList	list;
ViStatus	vierr;
int		i;

    found=0;

    if(!viSession_drm)
    {
        vierr=viOpenDefaultRM(&viSession_drm);
        if(vierr!=VI_SUCCESS)
	    return(vierr);
    }

    vierr=viFindRsrc(viSession_drm,"?*INSTR",&list,&num,descrip);
    if(vierr!=VI_SUCCESS)
        return(vierr);

    if(vxiident(descrip))
    {
	/* get first resource */
	if(found==0)
	{
	char *cp;
	    (void) strncpy(rsrc,descrip,rsrcLen);
            cp=strpbrk(rsrc,":");
            if(!cp)
	        rsrc[0]='\0';
            else
	        *cp='\0';
	}
	/* get LA */
	if(found<listSize)
	{
	    *LAlist=getLA(descrip);
	    LAlist++;
	    found++;
	    *numFound=found;
	}
    }

    for(i=0;i<(long)num-1;i++)
    {
        vierr=viFindNext(list,descrip);
        if(vierr!=VI_SUCCESS)
	{
            viClose(list);
	    return(vierr);
	}
        if(vxiident(descrip))
        {
	    /* get first resource */
	    if(found==0)
	    {
	        char *cp;
                (void) strncpy(rsrc,descrip,rsrcLen);
                cp=strpbrk(rsrc,":");
                if(!cp)
	            rsrc[0]='\0';
                else
	            *cp='\0';
	    }
	    /* get LA */
	    if(found<listSize)
	    {
	        *LAlist=getLA(descrip);
	        LAlist++;
	        found++;
	        *numFound=found;
	    }
        }
    }
    viClose(list);
    return 0;
}

SHORTSIZ16
i1432_visa_setup(E1432_MODULE_LIST_NODE *mn)
{
static     char    errstr[80]; /* static for VTL error message */
#if 0
    ViInt32 d32test;
#endif

    SHORTSIZ16 error;

    mn->a16_base = 0;
    i1432_error_info = NULL;	/* Not needed? */

    /* Default to "VXI" if the user never called e1432_set_interface_addr */
    if (strlen(i1432_sicl_name) == 0)
        (void) strcpy(i1432_sicl_name, "VXI");

    /* this sets the mn->a24_256k flag among others */
    if (!i1432_sema_present(mn))
	return i1432_la_print_error(mn->la, ERR1432_NO_MOD_AT_LA);

    /* set these lower level sicl addresses to NULL for now */
                               /* point to 128K to use the movable window */
    mn->a24_base = (LONGSIZ32 *)0x20000;
    if(mn->a24_256k)
        mn->a24_fifo_base = (LONGSIZ32 *)0x10000;
    else
        mn->a24_fifo_base = (LONGSIZ32 *)0x20000;

#ifdef	NO_D32
    mn->d32 = 0;
#else
    mn->d32 = 1;
#endif

    /* Now, just in case i1432_visa_d16 is 1 from a previous use */
    if (i1432_visa_d16)
        mn->d32=0;

    /* Write to Page Map register, to point to B Bus */
    /* this maps the top 128K of A24 to overlap the bottom 128K */
    /* all non data fifo writes and reads need to use the 128K map */
    error = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    if (error)
	return error;

    /* Enable local bus */
    error = i1432_direct_write_register(mn, E1432_PORT_CTL_REG,
					E1432_LBUS_ENABLE);
    if (error)
	return error;

#if 0
    /* try a 32 bit read to test the interface for 32 bit capability */
    /* the d32 bit will be reset to 0 if d32 fails */
    error = i1432_direct_read32_register(mn, E1432_FW_MARKER_REG, &d32test);
    if (error)
	return error;


    if (!i1432_sema_present(mn))
	return i1432_la_print_error(mn->la, ERR1432_NO_MOD_AT_LA);
#endif


    return(0);
}


SHORTSIZ16
i1432_fake_setup_sicl(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 la,
		      struct i1432_chan_info *chan_list_ptr,
		      int extra_io)
{
    SHORTSIZ16 error;

    /* Default to "VXI" if the user never called e1432_set_interface_addr */
    if (strlen(i1432_sicl_name) == 0)
        (void) strcpy(i1432_sicl_name, "VXI");

    extra_io; /* to stop a compiler warning */
    /* Clean up previous stuff, if needed */
    mn->la = la;
    error = i1432_visa_cleanup(mn);
    if (error)
	return error;

    /* Fake out some library functions into thinking we already did an
       e1432_assign_channel_numbers */
    mn->la = la;
    chan_list_ptr->mn = mn;
    i1432_chan_list[0] = chan_list_ptr;
    i1432_chan_count[0] = 1;

    error = i1432_visa_setup(mn);
    if (error)
	return error;

    return 0;
}


SHORTSIZ16
i1432_visa_cleanup(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 err;

    /* connect, so we can disconnect, eh? */
    err=i1432_module_connect(mn);

#if 0 /* it is ok to fail here */
    if(err)
	return(err);
#endif

    if(openSessions[mn->la].open)
    {
	if (i1432_visa_lock_mode)
	    /* Unlock the session first */
	    (void) viUnlock(openSessions[mn->la].sessionID);

        openSessions[mn->la].open = 0;
        viClose((ViSession)(mn->sicl_id));
        openSessions[mn->la].sessionID=0;
        mn->sicl_id=0; /* although no code should be checking sicl_id */
        NumOpenViSessions--;
    }

    return 0;
}




SHORTSIZ16
i1432_fake_cleanup_sicl(E1432_MODULE_LIST_NODE *mn,
			struct i1432_chan_info *chan_list_ptr,
			int chan_count)
{
    i1432_chan_list[0] = chan_list_ptr;
    i1432_chan_count[0] = chan_count;

    i1432_visa_cleanup(mn); /* don't want to hear about an error here */

    return 0;
}



SHORTSIZ16
i1432_calc_register_address(E1432_MODULE_LIST_NODE *mn,
			    LONGSIZ32 reg,
			    volatile SHORTSIZ16 ** regAddrPtr)
{
    *regAddrPtr = (volatile SHORTSIZ16 *) reg;
    return 0;
}


char *igeterrstr (int errnum)
{
static char s[]="unimplemented";
return s;
}

/*
 *********************************************************************
 This is the equivalent of SICL's iintroff.
 *********************************************************************
 */
SHORTSIZ16
i1432_visa_introff(void)
{
    ViStatus status;
    int     i;

    if (i1432_visa_introff_count == 0)
	for (i = 1; i < 256; i++)
	    if (openSessions[i].open)
	    {
		status = viEnableEvent(openSessions[i].sessionID,
				       VI_ALL_ENABLED_EVENTS,
				       VI_SUSPEND_HNDLR, VI_NULL);
		/* NI VISA on HP-UX returns invalid event if no events
		   are currently handled, so allow
		   VI_ERROR_INV_EVENT.  HP VISA on HP-UX returns
		   VI_ERROR_HANDLER_NINSTALLED in the same situation,
		   so allow that too. */
		if (status < VI_SUCCESS &&
		    status != VI_ERROR_INV_EVENT &&
		    status != VI_ERROR_HNDLR_NINSTALLED)
		    return
			i1432_print_error(convert_error_VTL_to_SICL(status));
	    }

    i1432_visa_introff_count++;
    return 0;
}

/*
 *********************************************************************
 This is the equivalent of SICL's iintron.
 *********************************************************************
 */
SHORTSIZ16
i1432_visa_intron(void)
{
    ViStatus status;
    int     i;

    i1432_visa_introff_count--;

    if (i1432_visa_introff_count == 0)
	for (i = 1; i < 256; i++)
	    if (openSessions[i].open)
	    {
		status = viEnableEvent(openSessions[i].sessionID,
				       VI_ALL_ENABLED_EVENTS,
				       VI_HNDLR, VI_NULL);
		/* NI VISA on HP-UX returns invalid event if no events
		   are currently handled, so allow
		   VI_ERROR_INV_EVENT.  HP VISA on HP-UX returns
		   VI_ERROR_HANDLER_NINSTALLED in the same situation,
		   so allow that too. */
		if (status < VI_SUCCESS &&
		    status != VI_ERROR_INV_EVENT &&
		    status != VI_ERROR_HNDLR_NINSTALLED)
		    return
			i1432_print_error(convert_error_VTL_to_SICL(status));
	    }

    return 0;
}

/*********************************************************************
*
* low level transfers
*
***********************************************************************/

/* the basic block move */
static int move_l_from_NT(INST id, unsigned long *src, unsigned long *dest,
		      unsigned long cnt, int swap)
{
ViStatus	vierr;

    /* try 32 bit first, then revert to 16 bit moves */
    do
    {
	if (!i1432_visa_d16)
	{
            vierr=viMoveIn32((ViSession)id,
                              VI_A24_SPACE,
                              (ViBusAddress)src,
		              (ViBusSize)cnt,
		              (ViUInt32 *)dest);
	    if((vierr==VI_ERROR_NSUP_WIDTH)|| (vierr==VI_ERROR_TMO))
	    {
		i1432_visa_d16 = 1;
		continue;
	    }
	    else
    	        return convert_error_VTL_to_SICL(vierr);
	}
	else /* try 16 bit transfer */
	{
            vierr=viMoveIn16((ViSession)id,
                              VI_A24_SPACE,
                              (ViBusAddress)src,
		              (ViBusSize)cnt*2,
		              (ViUInt16 *)dest);
    	    return convert_error_VTL_to_SICL(vierr);
	}
    } while (1);

    return VI_SUCCESS; /* won't get here */
}


/* size limited block moves, helps NI NT problem */
static int move_l_from_BLOCK(INST id, unsigned long *src, unsigned long *dest,
		         unsigned long cnt, int swap)
{
#ifdef FIX_NI_VIMOVE /* temporary patch for blocksize problem */
ViStatus vierr;
unsigned long c;

    while(cnt>0)
    {
	/* clip c at MAX_NI_BLOCK */
	c=(cnt>MAX_NI_BLOCK)?MAX_NI_BLOCK:cnt;
        vierr=viMoveIn32((ViSession)id,
                          VI_A24_SPACE,
                          (ViBusAddress)src,
		          (ViBusSize)c,
		          (ViUInt32 *)dest);
        if(vierr)
            return convert_error_VTL_to_SICL(vierr);
	src  += c;
	dest += c;
	cnt  -= c;
    }

    return convert_error_VTL_to_SICL(vierr);
#else
    return -1;
#endif
}

#ifdef	_HPUX_SOURCE
/* This function is used only on HP-UX, and only for D32 block
   transfers.  The possible interfaces I know of are:
	MXI-1
	    This interface normally can only do D16, even when you ask
	    it to do a 32-bit transfer.  There is a special mode to
	    get it to really do D32, but VISA does not provide a way
	    to get this mode.  From my testing, pointer access is
	    about the same speed as using VISA's viMoveIn32 or
	    viMoveIn16, probably because the bottleneck is the MXI
	    interface transfer speed.
	V/743
	    This can do real D32.  I believe pointer access is faster
	    than using viMoveIn32.  Pointer access *is* faster when
	    using SICL, and VISA is built on SICL.  Also, Keith claims
	    this is the fastest way on HP-UX.
	MXI-2
	    This can do real D32.  The VISA for this is provided by
	    NI.  I have no idea whether pointer access is fastest, but
	    on NI VISA on HP-UX, pointer access ends up byte-swapped.
	    Also, I think I saw occasional problems with viMapAddress
	    on this platform.  The right way to do things is probably
	    to use viMoveIn32 anyway.
   So the optimal method depends on the interface (MXI1 = don't care,
   V/743 = ptr is faster, MXI2 = ptr is broken/viMove is faster).
   Yuk. */

/* pointer moves for HP-UX */
static int move_l_from_HPUX(INST id, unsigned long *src, unsigned long *dest,
			    unsigned long cnt, int swap)
{
#if 1
    /* Use viMove for everything, so won't get byte-swapping on NI
       VISA on HP-UX. */
    return move_l_from_NT(id, src, dest, cnt, swap);
#else
ViStatus	vierr;
ViUInt32 	*mapadr;
ViUInt32	*to;	
ViUInt16	access;
ViUInt32	i;
	
    vierr=viMapAddress((ViSession)id,
		       VI_A24_SPACE,
		       (ViBusAddress)src,
		       (ViBusSize)cnt*4,
                       VI_FALSE,
		       VI_NULL,
		       (ViAddr)&mapadr);


    /* try to map a pointer, pointers are faster than viMove on HP-UX */
    if(!vierr) /* imap worked */
    {

        /* see if we have pointer access */
        vierr=viGetAttribute((ViSession)id,VI_ATTR_WIN_ACCESS,&access);
        if(vierr)
	    return ERR1432_UNABLE_TO_MAP_MEMORY;

        to=dest;
        if(access==VI_DEREF_ADDR)
        {


            for (i = 0; i < cnt % 16; i++)
            {
	        *(to++)=   *(mapadr++);
            }
            cnt /= 16;
            for (i = 0; i < cnt; i++)
            {
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
	        *(to++)=   *(mapadr++);
            }
    
        }
        else if(access==VI_USE_OPERS)
        {
	    /* just use block move */
            vierr=viUnmapAddress((ViSession)id);
            if(vierr)
	        return ERR1432_UNABLE_TO_MAP_MEMORY;
	    return move_l_from_NT(id, src, dest, cnt, swap);
        }
        else 
	    return ERR1432_UNABLE_TO_MAP_MEMORY;

        vierr=viUnmapAddress((ViSession)id);
        if(vierr)
	    return ERR1432_UNABLE_TO_MAP_MEMORY;
    }
    else  /* can't map address, so use highlevel transfer */
    {

        return move_l_from_NT(id, src, dest, cnt, swap);

    }

    return 0;

#endif
}
#endif


/* this is only for reading FROM the e1432 */
int ilblockcopy(INST id, unsigned long *src, unsigned long *dest,
                unsigned long cnt, int swap)
{
#ifdef _HPUX_SOURCE

if(pnptracelevel&2)
printf("vtlio: ilblockcopy(src=%p, dest=%p, cnt=%lu, swap=%d)\n",
	 src,dest,cnt,swap);


    return move_l_from_HPUX(id, src, dest, cnt, swap);

#else

#ifdef FIX_NI_VIMOVE /* temporary patch for blocksize problem */

    return move_l_from_BLOCK(id, src, dest, cnt, swap);

#else

    return move_l_from_NT(id, src, dest, cnt, swap);

#endif

#endif

}


/* this is for WRITING TO the e1432 */
int ilblockcopyout(INST id, unsigned long *src, unsigned long *dest,
                unsigned long cnt, int swap)
{
ViStatus	vierr;

if(pnptracelevel&2)
printf("vtlio: ilblockcopyout(src=%p, dest=%p, cnt=%lu, swap=%d)\n",
	 src,dest,cnt,swap);

    /* try 32 bit first, then revert to 16 bit moves */
    do
    {
	if (!i1432_visa_d16)
	{
            vierr=viMoveOut32((ViSession)id,
                              VI_A24_SPACE,
                              (ViBusAddress)dest,
		              (ViBusSize)cnt,
		              (ViUInt32 *)src);
	    if((vierr==VI_ERROR_NSUP_WIDTH)|| (vierr==VI_ERROR_TMO))
	    {
		i1432_visa_d16 = 1;
		continue;
	    }
	    else
    	        return convert_error_VTL_to_SICL(vierr);
	}
	else /* try 16 bit transfer */
	{
            vierr=viMoveOut16((ViSession)id,
                              VI_A24_SPACE,
                              (ViBusAddress)dest,
		              (ViBusSize)cnt*2,
		              (ViUInt16 *)src);
    	    return convert_error_VTL_to_SICL(vierr);
	}
    } while (1);

    return VI_SUCCESS; /* won't get here */
}


/* this is only for reading FROM the e1432 */
int iwblockcopy(INST id, unsigned short *src, unsigned short *dest,
                unsigned long cnt, int swap)
{
#ifdef FIX_NI_VIMOVE /* temporary patch for blocksize problem */
ViStatus vierr;
unsigned long c;

if(pnptracelevel&2)
printf("vtlio: iwblockcopy(src=%p, dest=%p, cnt=%lu, swap=%d)\n",
	 src,dest,cnt,swap);

    while(cnt>0)
    {
	/* clip c at MAX_NI_BLOCK */
	c=(cnt>MAX_NI_BLOCK)?MAX_NI_BLOCK:cnt;
        vierr=viMoveIn16((ViSession)id,
                          VI_A24_SPACE,
                          (ViBusAddress)src,
		          (ViBusSize)c,
		          (ViUInt16 *)dest);
        if(vierr)
            return convert_error_VTL_to_SICL(vierr);
	src  += c;
	dest += c;
	cnt  -= c;
    }

    return convert_error_VTL_to_SICL(vierr);


#else
ViStatus	vierr;

    vierr=viMoveIn16((ViSession)id,
                      VI_A24_SPACE,
                      (ViBusAddress)src,
		      (ViBusSize)cnt,
		      (ViUInt16 *)dest);

    return convert_error_VTL_to_SICL(vierr);
#endif
}


/* this is for WRITING TO the e1432 */
int iwblockcopyout(INST id, unsigned short *src, unsigned short *dest,
                unsigned long cnt, int swap)
{
ViStatus	vierr;

if(pnptracelevel&2)
printf("vtlio: iwblockcopyout(src=%p, dest=%p, cnt=%lu, swap=%d)\n",
	 src,dest,cnt,swap);

    vierr=viMoveOut16((ViSession)id,
                      VI_A24_SPACE,
                      (ViBusAddress)dest,
		      (ViBusSize)cnt,
		      (ViUInt16 *)src);

    return convert_error_VTL_to_SICL(vierr);
}


int iwpushfifo(INST id, unsigned short *src,unsigned short *fifo,
               unsigned long cnt, int swap)
{
ViStatus	vierr;


    vierr=viSetAttribute((ViSession)id,VI_ATTR_DEST_INCREMENT,(ViAttrState)0);
    if(vierr)
	return ERR1432_UNABLE_TO_MAP_MEMORY;

    vierr=viMoveOut16((ViSession)id,
                      VI_A24_SPACE,
                      (ViBusAddress)fifo,
		      (ViBusSize)cnt,
		      (ViUInt16 *)src);
    if(vierr)
        return convert_error_VTL_to_SICL(vierr);

    vierr=viSetAttribute((ViSession)id,VI_ATTR_DEST_INCREMENT,(ViAttrState)1);

    return convert_error_VTL_to_SICL(vierr);


}


SHORTSIZ16 i1432_direct_write_register(E1432_MODULE_LIST_NODE *mn,
                                              LONGSIZ32 reg, SHORTSIZ16 data)
{
ViStatus	rv;
ViBusAddress	addr;
SHORTSIZ16 	err, err2;

    err = i1432_introff();
    if (err)
	return err;

    err=i1432_module_connect(mn);
    if(err)
	goto cleanup;

    if(reg>= 0x40)
	addr=(long)(mn->a24_base)+(long)reg;
    else
	addr=reg;

if(pnptracelevel&2)
printf("vtlio: direct_write_register %lx, addr=%p,  data=%x\n",reg,addr,data);


    rv=viOut16((ViSession)(mn->sicl_id),VI_A24_SPACE, addr, data);

 cleanup:
    err2 = i1432_intron();
    if (err2)
	return err2;

    if (err)
	return err;

    return convert_error_VTL_to_SICL(rv);
}


SHORTSIZ16 i1432_direct_read_register(E1432_MODULE_LIST_NODE *mn,
                                             LONGSIZ32 reg, SHORTSIZ16 *data)
{
ViUInt16	readval;
ViStatus	rv;
ViBusAddress	addr;
SHORTSIZ16 	err, err2;

    err = i1432_introff();
    if (err)
	return err;

    err=i1432_module_connect(mn);
    if(err)
	goto cleanup;

    if(reg>= 0x40)
	addr=(long)(mn->a24_base)+(long)reg;
    else
	addr=reg;

    rv=viIn16((ViSession)(mn->sicl_id),VI_A24_SPACE, addr, &readval);
    *data=(SHORTSIZ16)readval;

if(pnptracelevel&2)
printf("vtlio: direct_read_register %lx, addr=%p, read=%x\n",reg,addr,*data);

 cleanup:
    err2 = i1432_intron();
    if (err2)
	return err2;

    if (err)
	return err;
    return convert_error_VTL_to_SICL(rv);
}

SHORTSIZ16 i1432_direct_write32_register_a(E1432_MODULE_LIST_NODE *mn,
			      LONGSIZ32 reg, int a24, LONGSIZ32 data)
{
ViStatus	vierr;
ViBusAddress	addr;
SHORTSIZ16 	rv, err2;

    rv = i1432_introff();
    if (rv)
	return rv;

    rv = i1432_module_connect(mn);
    if (rv)
	goto cleanup;

    if(a24)
	addr=(long)(mn->a24_base)+(long)reg;
    else
	addr=reg;

if(pnptracelevel&2)
printf("vtlio: direct_write32_register %lx, addr=%p, data=%lx\n",reg,addr,data);


    /* try 32 bit first, then revert to 16 bit moves */
    do
    {
	if (mn->d32 && !i1432_visa_d16)
	{
            vierr=viOut32((ViSession)(mn->sicl_id),VI_A24_SPACE, addr,
		                   (ViUInt32)(data));
	    if((vierr==VI_ERROR_NSUP_WIDTH)|| (vierr==VI_ERROR_TMO))
	    {
		i1432_visa_d16 = 1;
		mn->d32=0; /* turn off d32 access */
		continue;
	    }
	    else
	    {
    	        rv = convert_error_VTL_to_SICL(vierr);
		goto cleanup;
	    }
	}
	else /* try 16 bit transfer */
	{
            vierr=viOut16((ViSession)(mn->sicl_id),VI_A24_SPACE, addr,
		                   (ViUInt16)(data>>16)); /* write high */
            if(!vierr)
            vierr=viOut16((ViSession)(mn->sicl_id),VI_A24_SPACE, addr+2,
				   (ViUInt16)(data&0xffff));/* write low */
    	    rv = convert_error_VTL_to_SICL(vierr);
	    goto cleanup;
	}
    } while (1);

 cleanup:
    err2 = i1432_intron();
    if (err2)
	return err2;

    return rv;
}


SHORTSIZ16 i1432_direct_write32_register(E1432_MODULE_LIST_NODE *mn,
                                                LONGSIZ32 reg,
                                                LONGSIZ32 data)
{
    return i1432_direct_write32_register_a(mn, reg, (reg>= 0x40), data);
}

SHORTSIZ16 i1432_direct_read32_register_a(E1432_MODULE_LIST_NODE *mn,
			     LONGSIZ32 reg, int a24, LONGSIZ32 * data)
{
ViUInt16	first;
ViUInt32	second;
ViStatus	vierr;
ViBusAddress	addr;
SHORTSIZ16 	err2, rv;

    rv = i1432_introff();
    if (rv)
	return rv;

    rv = i1432_module_connect(mn);
    if (rv)
	goto cleanup;

    if(a24)
	addr=(long)(mn->a24_base)+(long)reg;
    else
	addr=reg;

    /* try 32 bit first, then revert to 16 bit moves */
    do
    {
	if (mn->d32 && !i1432_visa_d16)
	{
            vierr=viIn32((ViSession)(mn->sicl_id),VI_A24_SPACE, addr,
				     (ViPUInt32) data);
	    if((vierr==VI_ERROR_NSUP_WIDTH)|| (vierr==VI_ERROR_TMO))
	    {
		i1432_visa_d16 = 1;
		mn->d32=0; /* turn off d32 access */
		continue;
	    }
	    else
	    {
if(pnptracelevel&2)
printf("vtlio: direct_read32_register %lx, addr=%p, read=%x\n",reg,addr,*data);
    	        rv = convert_error_VTL_to_SICL(vierr);
		goto cleanup;
	    }
	}
	else /* try 16 bit transfer */
	{
            vierr=viIn16((ViSession)(mn->sicl_id),VI_A24_SPACE, addr, &first);
            if(!vierr) 
            {
	        second=first<<16;
                vierr=viIn16((ViSession)(mn->sicl_id),VI_A24_SPACE, addr+2,&first);
                if(!vierr) 
                {
                    second=second+first;
	            *data=(LONGSIZ32)second;
	        }
            }
if(pnptracelevel&2)
printf("vtlio: direct_read32_register %lx, addr=%p, read=%x\n",reg,addr,*data);
	    rv = convert_error_VTL_to_SICL(vierr);
	    goto cleanup;
	}
    } while (1);

 cleanup:
    err2 = i1432_intron();
    if (err2)
	return err2;

    return rv;
}

SHORTSIZ16 i1432_direct_read32_register(E1432_MODULE_LIST_NODE *mn,
                                               LONGSIZ32 reg,
                                               LONGSIZ32 *data)
{
    return i1432_direct_read32_register_a(mn, reg, (reg>= 0x40), data);
}

/* emulate the ibeswap function in sicl, other modules call this */
/* and the PNP build doesn't link with sicl library */
/* remember that this code only runs on the PC (intel) */

int ibeswap(char *addr,unsigned long length,int datasize)
{
    unsigned long i;
    char	temp;

    if(datasize==2)
    {
        for(i=0;i<length/2;i++)
	{
	    temp=addr[2*i];
	    addr[2*i]=addr[2*i+1];
	    addr[2*i+1]=temp;
	}
    }
    else if(datasize==4)
    {
        for(i=0;i<length/4;i++)
	{
	    temp=addr[4*i];
	    addr[4*i]=addr[4*i+3];
	    addr[4*i+3]=temp;
	    temp=addr[4*i+1];
	    addr[4*i+1]=addr[4*i+2];
	    addr[4*i+2]=temp;
	}
    }

    return 0;
}




/* call back */

#ifdef INSTR_CALLBACKS

static ViStatus _VI_FUNCH my_handler(ViSession vi,ViEventType eventType,
		                      ViEvent context, ViAddr usrHandle)
{
SHORTSIZ16	serror;
E1432_MODULE_LIST_NODE *mynode;
hpe1432_callbackFunc	func;

    /* determine module from channel */
    serror=i1432_get_module_from_chan(0, (SHORTSIZ16)usrHandle, &mynode);
    if(serror)
	return VI_SUCCESS;

    func=(hpe1432_callbackFunc)(mynode->cbFunc);

    /* call user handler if available */
    if(func)
    {
	(func)((ViInt32)(mynode->cbValue),(ViInt32)(usrHandle));
    }

    return VI_SUCCESS;
}

LONGSIZ32 
vtl_install_callback  (LONGSIZ32 vi, LONGSIZ32 channel,
			  void (*cbf)(), LONGSIZ32 value)
{
SHORTSIZ16	serror;
SHORTSIZ16	mod;
E1432_MODULE_LIST_NODE *mynode;
E1432_MODULE_LIST_NODE *mn;
int		want_handler;
ViStatus	vierror;

    /* determine module from channel */
    serror=i1432_get_module_from_chan(0, (SHORTSIZ16)channel, &mynode);
    if(serror)
	return VI_ERROR_PARAMETER2;

    /* store function pointer or null in cbfunc array */
    mynode->cbFunc=cbf;
    mynode->cbValue=value;
    mynode->cbHandle=channel; /* save for the uninstall */

/* install my_handler if there are any cbfuncs in callback array */
/* uninstall my_handler if no cbfuncs in callback array */
    want_handler=0;
    mn = i1432_mod_list;
    for (mod = 0; mod < i1432_mod_count; mod++, mn++)
    {
	if(mn->cbFunc)
	{
	    want_handler=1;
	    break;
	}
    }

#if 1
   /*
    * This part is added to fix LMS's problem that was caused by interrupt handler installed
    * multiple times.    This will remove any previously installed handler before installing
    * the new one.                                  Feb 13, 2000    ... Yuji Kobayashi
    */
    {
	vierror=viDisableEvent(mynode->sicl_id, VI_EVENT_VXI_SIGP, VI_HNDLR);
	if(  (vierror==VI_SUCCESS_EVENT_DIS)
	   ||(vierror==VI_ERROR_INV_EVENT))
	    vierror=VI_SUCCESS;
	if(vierror)
	{
            i1432_set_error_info("viDisableEvent failed");
	    return vierror;
	}
	vierror=viUninstallHandler(mynode->sicl_id, VI_EVENT_VXI_SIGP,
				   VI_ANY_HNDLR, (ViAddr) (mynode->cbHandle));
	/* no harm done if error here */
	if(vierror)
	    vierror=VI_SUCCESS;
    }
#endif


    if(want_handler)
    {
	vierror=viInstallHandler(mynode->sicl_id,VI_EVENT_VXI_SIGP,
				 (ViHndlr)(my_handler),(ViAddr)channel);
	if(vierror)
	{
            i1432_set_error_info("viInstallHandler failed");
	    return vierror;
	}
	if (i1432_visa_introff_count > 0)
	    vierror=viEnableEvent(mynode->sicl_id, VI_EVENT_VXI_SIGP,
				  VI_SUSPEND_HNDLR, VI_NULL);
	else
	    vierror=viEnableEvent(mynode->sicl_id, VI_EVENT_VXI_SIGP,
				  VI_HNDLR, VI_NULL);
	if(vierror)
	{
            i1432_set_error_info("viEnableEvent failed");
	    return vierror;
	}

    }
#if 0
   /*
    * This part is removed because interrupt handler is now removed always.
    *        Feb 13, 2000                     Yuji Kobayashi
    */
    else
    {
	vierror=viDisableEvent(mynode->sicl_id, VI_EVENT_VXI_SIGP, VI_HNDLR);
	if(  (vierror==VI_SUCCESS_EVENT_DIS)
	   ||(vierror==VI_ERROR_INV_EVENT))
	    vierror=VI_SUCCESS;
	if(vierror)
	{
            i1432_set_error_info("viDisableEvent failed");
	    return vierror;
	}
	vierror=viUninstallHandler(mynode->sicl_id, VI_EVENT_VXI_SIGP,
				   VI_ANY_HNDLR, (ViAddr) (mynode->cbHandle));
	/* no harm done if error here */
	if(vierror)
	    vierror=VI_SUCCESS;
    }
#endif

    return VI_SUCCESS;
}


LONGSIZ32 vtl_remove_callbacks(void)
{
SHORTSIZ16	mod;
E1432_MODULE_LIST_NODE *mn;
ViStatus	vierror;

    /* walk thru module array and uninstall handlers */
    mn = i1432_mod_list;
    for (mod = 0; mod < i1432_mod_count; mod++, mn++)
    {
	if(mn->cbFunc)
	{
	    vierror=viDisableEvent(mn->sicl_id, VI_EVENT_VXI_SIGP,
				  VI_HNDLR);
            /* viDisableEvent returns VI_SUCCESS_EVENT_DIS if already off */
	   if(  (vierror==VI_SUCCESS_EVENT_DIS)
	      ||(vierror==VI_ERROR_INV_EVENT))
		vierror=VI_SUCCESS;

	    if(vierror)
	    {
                i1432_set_error_info("viDisableEvent failed");
	        return vierror;
	    }
	    vierror=viUninstallHandler(mn->sicl_id,VI_EVENT_VXI_SIGP,
				 VI_ANY_HNDLR,(ViAddr)(mn->cbHandle));
	    /* no harm done if error here */
	    if(vierror)
	        vierror=VI_SUCCESS;
	}
    }

    return VI_SUCCESS;
}

#endif /* INSTR_CALLBACKS */



#endif /* HAVE_VTL  this file is only used for VTL */


/* this next line is here to stop a silly compiler warning for non PLUG&PLAY */
extern int hpplug_quiet;








